<?php
/**
 * Plugin Name: WordPress AI Editor Bridge
 * Description: Enhanced REST API endpoints for BulkForge - Provides advanced post editing, SEO integration, and revision management
 * Version: 1.0.0
 * Author: BulkForge
 * Author URI: https://bulkforge.ai
 * License: GPL v3 or later
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 * Text Domain: wp-ai-editor-bridge
 * Domain Path: /languages
 * Requires: 5.9
 * Requires PHP: 7.4
 *
 * This plugin provides enhanced REST API endpoints at /wp-json/wpai/v1/ for BulkForge integration.
 * It enables features like direct Yoast SEO/RankMath integration, revision management, and bulk operations.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * BulkForge AI Editor Bridge Main Class
 */
class WP_AI_Editor_Bridge {

	const VERSION = '1.0.0';
	const NAMESPACE = 'wpai/v1';
	const PLUGIN_NAME = 'wp-ai-editor-bridge';

	/**
	 * Initialize the plugin
	 */
	public static function init() {
		$instance = new self();
		add_action( 'rest_api_init', array( $instance, 'register_routes' ) );
		add_action( 'init', array( $instance, 'load_textdomain' ) );
	}

	/**
	 * Load plugin text domain for translations
	 */
	public function load_textdomain() {
		load_plugin_textdomain( self::PLUGIN_NAME, false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
	}

	/**
	 * Register REST API routes
	 */
	public function register_routes() {
		// Capabilities endpoint - Detect what the plugin supports
		register_rest_route(
			self::NAMESPACE,
			'/capabilities',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_capabilities' ),
				'permission_callback' => array( $this, 'check_permission' ),
			)
		);

		// SEO Meta endpoint - Update Yoast/RankMath fields
		register_rest_route(
			self::NAMESPACE,
			'/seo-meta/(?P<id>\d+)',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_seo_meta' ),
				'permission_callback' => array( $this, 'check_permission' ),
				'args'                => array(
					'id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			)
		);

		// Schema/Structured Data endpoint
		register_rest_route(
			self::NAMESPACE,
			'/schema/(?P<id>\d+)',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_schema' ),
				'permission_callback' => array( $this, 'check_permission' ),
				'args'                => array(
					'id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			)
		);

		// Revisions endpoint - List post revisions
		register_rest_route(
			self::NAMESPACE,
			'/revisions/(?P<id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_revisions' ),
				'permission_callback' => array( $this, 'check_permission' ),
				'args'                => array(
					'id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			)
		);

		// Create Revision endpoint
		register_rest_route(
			self::NAMESPACE,
			'/revision/(?P<id>\d+)',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'create_revision' ),
				'permission_callback' => array( $this, 'check_permission' ),
				'args'                => array(
					'id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			)
		);

		// Rollback Revision endpoint - Restore a previous revision
		register_rest_route(
			self::NAMESPACE,
			'/rollback/(?P<id>\d+)',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'rollback_revision' ),
				'permission_callback' => array( $this, 'check_permission' ),
				'args'                => array(
					'id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			)
		);

		// Bulk Update endpoint - Update multiple posts at once
		register_rest_route(
			self::NAMESPACE,
			'/bulk-update',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'bulk_update_posts' ),
				'permission_callback' => array( $this, 'check_permission' ),
			)
		);

		// Custom Fields endpoint
		register_rest_route(
			self::NAMESPACE,
			'/custom-fields/(?P<id>\d+)',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_custom_fields' ),
				'permission_callback' => array( $this, 'check_permission' ),
				'args'                => array(
					'id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			)
		);

		// Cache clear endpoint
		register_rest_route(
			self::NAMESPACE,
			'/cache/clear/(?P<id>\d+)',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'clear_post_cache' ),
				'permission_callback' => array( $this, 'check_permission' ),
				'args'                => array(
					'id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			)
		);
	}

	/**
	 * Check if user has permission to use the API
	 *
	 * @param WP_REST_Request $request The request object
	 * @return boolean
	 */
	public function check_permission( $request ) {
		return current_user_can( 'edit_posts' );
	}

	/**
	 * Get plugin capabilities
	 *
	 * @param WP_REST_Request $request The request object
	 * @return WP_REST_Response
	 */
	public function get_capabilities( $request ) {
		$capabilities = array(
			'plugin_active'      => true,
			'plugin_version'     => self::VERSION,
			'features'           => array(
				'seo_meta'       => $this->has_seo_plugin(),
				'schema'         => true,
				'revisions'      => true,
				'bulk_update'    => true,
				'custom_fields'  => true,
				'cache_clearing' => true,
			),
			'seo_plugin'         => $this->detect_seo_plugin(),
			'wordpress_version'  => get_bloginfo( 'version' ),
			'php_version'        => phpversion(),
		);

		return new WP_REST_Response( $capabilities, 200 );
	}

	/**
	 * Detect which SEO plugin is installed
	 *
	 * @return string|null Plugin name or null if none detected
	 */
	private function detect_seo_plugin() {
		if ( function_exists( 'YoastSEO' ) || class_exists( 'WPSEO_Options' ) ) {
			return 'yoast';
		}
		if ( function_exists( 'RankMath' ) || class_exists( 'RankMath\RankMath' ) ) {
			return 'rankmath';
		}
		return null;
	}

	/**
	 * Check if an SEO plugin is available
	 *
	 * @return boolean
	 */
	private function has_seo_plugin() {
		return ! is_null( $this->detect_seo_plugin() );
	}

	/**
	 * Update SEO meta data (Yoast/RankMath)
	 *
	 * @param WP_REST_Request $request The request object
	 * @return WP_REST_Response|WP_Error
	 */
	public function update_seo_meta( $request ) {
		$post_id = $request['id'];
		$data    = $request->get_json_params();

		// Verify post exists
		$post = get_post( $post_id );
		if ( ! $post ) {
			return new WP_Error( 'post_not_found', 'Post not found', array( 'status' => 404 ) );
		}

		$seo_plugin = $this->detect_seo_plugin();

		if ( 'yoast' === $seo_plugin ) {
			$this->update_yoast_meta( $post_id, $data );
		} elseif ( 'rankmath' === $seo_plugin ) {
			$this->update_rankmath_meta( $post_id, $data );
		} else {
			return new WP_Error( 'no_seo_plugin', 'No compatible SEO plugin found', array( 'status' => 400 ) );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'message' => 'SEO meta updated successfully',
				'post_id' => $post_id,
			),
			200
		);
	}

	/**
	 * Update Yoast SEO meta
	 *
	 * @param int   $post_id The post ID
	 * @param array $data The data to update
	 */
	private function update_yoast_meta( $post_id, $data ) {
		// Yoast meta fields
		if ( isset( $data['yoast_title'] ) ) {
			update_post_meta( $post_id, '_yoast_wpseo_title', sanitize_text_field( $data['yoast_title'] ) );
		}
		if ( isset( $data['yoast_description'] ) ) {
			update_post_meta( $post_id, '_yoast_wpseo_metadesc', sanitize_text_field( $data['yoast_description'] ) );
		}
		if ( isset( $data['yoast_focus_keyword'] ) ) {
			update_post_meta( $post_id, '_yoast_wpseo_focuskw', sanitize_text_field( $data['yoast_focus_keyword'] ) );
		}
	}

	/**
	 * Update RankMath SEO meta
	 *
	 * @param int   $post_id The post ID
	 * @param array $data The data to update
	 */
	private function update_rankmath_meta( $post_id, $data ) {
		// RankMath meta fields
		if ( isset( $data['rankmath_title'] ) ) {
			update_post_meta( $post_id, 'rank_math_title', sanitize_text_field( $data['rankmath_title'] ) );
		}
		if ( isset( $data['rankmath_description'] ) ) {
			update_post_meta( $post_id, 'rank_math_description', sanitize_text_field( $data['rankmath_description'] ) );
		}
		if ( isset( $data['rankmath_focus_keyword'] ) ) {
			update_post_meta( $post_id, 'rank_math_focus_keyword', sanitize_text_field( $data['rankmath_focus_keyword'] ) );
		}
	}

	/**
	 * Update schema/structured data
	 *
	 * @param WP_REST_Request $request The request object
	 * @return WP_REST_Response|WP_Error
	 */
	public function update_schema( $request ) {
		$post_id = $request['id'];
		$data    = $request->get_json_params();

		$post = get_post( $post_id );
		if ( ! $post ) {
			return new WP_Error( 'post_not_found', 'Post not found', array( 'status' => 404 ) );
		}

		if ( isset( $data['schema_json'] ) ) {
			// Validate JSON
			$schema = json_decode( $data['schema_json'], true );
			if ( json_last_error() !== JSON_ERROR_NONE ) {
				return new WP_Error( 'invalid_json', 'Invalid JSON in schema_json', array( 'status' => 400 ) );
			}

			update_post_meta( $post_id, '_bulkforge_schema', wp_json_encode( $schema ) );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'message' => 'Schema updated successfully',
				'post_id' => $post_id,
			),
			200
		);
	}

	/**
	 * Get post revisions
	 *
	 * @param WP_REST_Request $request The request object
	 * @return WP_REST_Response|WP_Error
	 */
	public function get_revisions( $request ) {
		$post_id = $request['id'];
		$post    = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error( 'post_not_found', 'Post not found', array( 'status' => 404 ) );
		}

		$revisions = wp_get_post_revisions( $post_id, array( 'check_restore_revision_permission' => false ) );

		$revision_data = array();
		foreach ( $revisions as $revision ) {
			$revision_data[] = array(
				'id'           => $revision->ID,
				'date'         => $revision->post_date,
				'modified'     => $revision->post_modified,
				'author'       => get_the_author_meta( 'display_name', $revision->post_author ),
				'title'        => $revision->post_title,
				'content'      => $revision->post_content,
				'excerpt'      => $revision->post_excerpt,
			);
		}

		return new WP_REST_Response(
			array(
				'revisions' => $revision_data,
				'count'     => count( $revision_data ),
			),
			200
		);
	}

	/**
	 * Create a revision of the current post
	 *
	 * @param WP_REST_Request $request The request object
	 * @return WP_REST_Response|WP_Error
	 */
	public function create_revision( $request ) {
		$post_id = $request['id'];
		$post    = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error( 'post_not_found', 'Post not found', array( 'status' => 404 ) );
		}

		// WordPress automatically creates revisions on post updates
		// This endpoint documents that revisions are enabled
		$revisions = wp_get_post_revisions( $post_id );

		return new WP_REST_Response(
			array(
				'success'    => true,
				'message'    => 'Revision system is active',
				'revision_count' => count( $revisions ),
			),
			200
		);
	}

	/**
	 * Rollback to a previous revision
	 *
	 * @param WP_REST_Request $request The request object
	 * @return WP_REST_Response|WP_Error
	 */
	public function rollback_revision( $request ) {
		$post_id      = $request['id'];
		$data         = $request->get_json_params();
		$revision_id  = isset( $data['revision_id'] ) ? absint( $data['revision_id'] ) : 0;

		if ( ! $revision_id ) {
			return new WP_Error( 'missing_revision_id', 'revision_id is required', array( 'status' => 400 ) );
		}

		$post     = get_post( $post_id );
		$revision = get_post( $revision_id );

		if ( ! $post || ! $revision ) {
			return new WP_Error( 'post_not_found', 'Post or revision not found', array( 'status' => 404 ) );
		}

		// Restore the revision
		$restored = wp_restore_post_revision( $revision_id );

		if ( ! $restored ) {
			return new WP_Error( 'restore_failed', 'Failed to restore revision', array( 'status' => 500 ) );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'message' => 'Post restored to revision',
				'post_id' => $post_id,
				'revision_id' => $revision_id,
			),
			200
		);
	}

	/**
	 * Bulk update multiple posts
	 *
	 * @param WP_REST_Request $request The request object
	 * @return WP_REST_Response|WP_Error
	 */
	public function bulk_update_posts( $request ) {
		$data = $request->get_json_params();

		if ( ! isset( $data['updates'] ) || ! is_array( $data['updates'] ) ) {
			return new WP_Error( 'missing_updates', 'updates array is required', array( 'status' => 400 ) );
		}

		$results = array();
		foreach ( $data['updates'] as $update ) {
			$post_id = absint( $update['id'] ?? 0 );
			if ( ! $post_id ) {
				continue;
			}

			$post_data = array(
				'ID' => $post_id,
			);

			if ( isset( $update['title'] ) ) {
				$post_data['post_title'] = sanitize_text_field( $update['title'] );
			}
			if ( isset( $update['content'] ) ) {
				$post_data['post_content'] = wp_kses_post( $update['content'] );
			}
			if ( isset( $update['excerpt'] ) ) {
				$post_data['post_excerpt'] = sanitize_text_field( $update['excerpt'] );
			}

			$updated = wp_update_post( $post_data );
			$results[] = array(
				'id'      => $post_id,
				'success' => is_numeric( $updated ) && ! is_wp_error( $updated ),
			);
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'updated_count' => count( $results ),
				'results' => $results,
			),
			200
		);
	}

	/**
	 * Update custom post fields
	 *
	 * @param WP_REST_Request $request The request object
	 * @return WP_REST_Response|WP_Error
	 */
	public function update_custom_fields( $request ) {
		$post_id = $request['id'];
		$data    = $request->get_json_params();

		$post = get_post( $post_id );
		if ( ! $post ) {
			return new WP_Error( 'post_not_found', 'Post not found', array( 'status' => 404 ) );
		}

		if ( ! isset( $data['fields'] ) || ! is_array( $data['fields'] ) ) {
			return new WP_Error( 'missing_fields', 'fields array is required', array( 'status' => 400 ) );
		}

		foreach ( $data['fields'] as $key => $value ) {
			// Sanitize key to prevent issues
			$key = sanitize_key( $key );
			if ( empty( $key ) ) {
				continue;
			}

			update_post_meta( $post_id, $key, wp_kses_post( $value ) );
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'message' => 'Custom fields updated',
				'post_id' => $post_id,
			),
			200
		);
	}

	/**
	 * Clear post cache
	 *
	 * @param WP_REST_Request $request The request object
	 * @return WP_REST_Response|WP_Error
	 */
	public function clear_post_cache( $request ) {
		$post_id = $request['id'];
		$post    = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error( 'post_not_found', 'Post not found', array( 'status' => 404 ) );
		}

		// Clear post-specific caches
		clean_post_cache( $post_id );

		// Clear term caches if associated
		wp_cache_delete( 'post_' . $post_id, 'posts' );

		// Flush full page cache if using cache plugins
		if ( function_exists( 'wp_cache_flush' ) ) {
			wp_cache_flush();
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'message' => 'Post cache cleared',
				'post_id' => $post_id,
			),
			200
		);
	}
}

// Initialize the plugin
WP_AI_Editor_Bridge::init();
